/**
 * WidgetData is a class for all component "models".
 *
 * This class is responsible for:
 * - Storing the Peer Node of the widget
 *
 * - Firing property updates to the view when a peer node property changes
 *
 * - Dispatch events to the peer node, when an event is observed from a WidgetView
 *
 * This class fires events in line with dojo/Evented conventions and can be observed with dojo/on.
 *
 * Name:        "childAdded"
 * Fired:       when a child WidgetData is added to this WidgetData
 * EventData:   {
 *                WidgetData : handle to the newly added WidgetData
 *              }
 *
 * Name:        "childRemoved"
 * Fired:       when a child WidgetData is removed from this WidgetData
 * EventData:   {
 *                  WidgetData : handle to the removed WidgetData
 *              }
 *
 * Name:        "propertiesSet"
 * Fired:       when a property changes
 * EventData:   [array of changed property names]
 *
 * Copyright 2014 MathWorks, Inc.
 */

define([
    "dojo/_base/declare",
    "dojo/Evented"
], function (declare, Evented) {
    return declare([Evented], {

        // Handle to the MW/peermodel/PeerNode that this WidgetData wraps
        _PeerNode: null,

        /**
         * Constructor
         *
         * @param {MW/peermodel/PeerNode} peerNode      The PeerNode that will serve as the model for this WidgetData
         */
        constructor: function (peerNode) {
            // Assertions
            if (!peerNode) {
                throw "PeerNode must be non empty / non null";
            }

            // Initialize Properties
            this._PeerNode = peerNode;

            // listen to property changes on Peer Node
            this._addListenersToPeerNode();
        },

        /**
         * Cleanup
         */
        destroy: function () {
            // unsubscribe from PeerNode events
            this._PeerNode.removeEventListener("propertiesSet", "_handlePeerNodePropertiesSet", this);
            this._PeerNode.removeEventListener("peerEvent", "_handlePeerNodePeerEvent", this);

            // remove reference
            delete this._PeerNode;
        },


        /**
         * Gets the Target domNode were the child widgets can be placed
         * @returns containerNode of the associated WidgetView
         */
        getTargetNode: function () {
            return this.WidgetView;
        },

        /**
         * Sets the WidgetView of the WidgetData
         */
        setTargetNode: function (widgetView) {
            this.WidgetView = widgetView;
        },

        /**
         * Gets a property for this WidgetData
         * @param propertyName - the name of the Property to get
         * @return The property value
         */
        getProperty: function (propertyName) {
            return this._PeerNode.getProperty(propertyName);
        },


        /**
         * Gets all properties defined for this peer node.
         * @returns An object containing all the peer node's properties
         */
        getProperties: function () {
            return this._PeerNode.getProperties();
        },


        /**
         * Sets a property for this WidgetData
         *
         * @param propertyName - the name of the Property to set
         * @param propertyValue - the value of the Property to set
         */
        setProperty: function (propertyName, propertyValue) {
            return this._PeerNode.setProperty(propertyName, propertyValue);
        },


        /**
         * Sets a group of properties for this WidgetData.
         *
         * @param properties Object containing the new properties.
         */
        setProperties: function (properties) {
            return this._PeerNode.setProperties(properties);
        },

        /**
         * Gets the ID of the WidgetData
         * @return a string representing the ID
         */
        getId: function () {
            return this._PeerNode.getId();
        },

        /**
         * Gets the Type of the WidgetData
         * @return a string representing the type
         */
        getType: function () {
            return this._PeerNode.getType();
        },

        /**
         * Adds the component
         * @param newWidgetData - the WidgetData of the component to be added
         */
        addChild: function (newWidgetData) {
            //Fire the widgetDataAdded event
            this.emit("widgetDataAdded", {
                WidgetData: newWidgetData
            });
        },

        /**
         * Removes component
         * @param removedWidgetData - the WidgetData of the component to be removed
         */
        removeChild: function (removedWidgetData) {
            //Fire the widgetDataRemoved event
            this.emit("widgetDataRemoved", {
                WidgetData: removedWidgetData
            });

        },

        /**
         * Tells the WidgetData to forward an event to the server
         *
         * @param eventName     the type of the event as a string
         *
         * @param eventData     an object holding the data needed by the server with fields such
         *                      as "Value", "Text", "NeedlePosition", "Theta", etc...
         */
        dispatchEventToServer: function (eventName, eventData) {
            // Put the eventName as a field of the data
            eventData.Name = eventName;
            // Forward the event through the peer node
            this._PeerNode.dispatchEvent("peerEvent", this._PeerNode, eventData);
        },

        /**
         * ------------------------------------------------------------------------------------------------------
         * Private Functions
         * ------------------------------------------------------------------------------------------------------
         */

        /**
         * Helper function which attaches listeners to the PeerNode
         */
        _addListenersToPeerNode: function () {
            // Note that this listens to "propertiesSet", which is fired when either a single property or multiple
            // properties changes, regardless if only one property or many were changed
            this._PeerNode.addEventListener("propertiesSet", "_handlePeerNodePropertiesSet", this);

            // listen to the "peerEvent" event
            this._PeerNode.addEventListener("peerEvent", "_handlePeerNodePeerEvent", this);
        },

        /**
         *  Helper function to handle "propertiesSet" event from Peer Node
         */
        _handlePeerNodePropertiesSet: function (event) {

            /**
             * For a "propertiesSet" event, the event has the following relevant field structure.
             *
             * - data
             * -- newValues
             * --- <propertyName>: "<new value for the corresponding property name >"
             * -- oldValues
             * --- <propertyName>: "<old value for the corresponding property name >"
             *
             * To create an object of key/value pairs, we can just pluck the "newValues"
             * contents directly out of the event and then get its keys
             **/
            var changedNameValuePairs = event.data.newValues;
            var changedPropertyNames = Object.keys(changedNameValuePairs);
            // fire
            this.emit("propertiesSet", changedPropertyNames);
        },

        /**
         *  Helper function to handle "peerEvent" event from Peer Node
         */
        _handlePeerNodePeerEvent: function (event) {
            var eventName = event.data.Name || event.data.eventType;
            // simply re-fire the event
            this.emit(eventName, event);
        }

    });
});
