/**
 * WidgetDataManager is responsible for
 *  - handling the Widget Data for the Web figure and Widget data
 *    of all the components under the Web figure.
 *  - observe the childAdded/childRemoved of the Web Figure PeerNode and
 *    all the components PeerNodes under Web Figure
 *
 * The WidgetDataManager requires a PeerNode holding the state of the Web figure and
 * is parent for the other components to be drawn
 *
 * This class fires events in line with dojo/Evented conventions and can be observed with dojo/on.
 *
 * Name:        "childAdded"
 * Fired:       when a child WidgetData is added to this WidgetData
 * EventData:   {
 *                WidgetData : handle to the newly added WidgetData
 *              }
 *
 * Name:        "childRemoved"
 * Fired:       when a child WidgetData is removed from this WidgetData
 * EventData:   {
 *                  WidgetData : handle to the removed WidgetData
 *              }
 *
 * Copyright 2014 MathWorks, Inc.
 */

define([
    "dojo/_base/declare",
    "dojo/_base/lang",
    "dojo/Evented",
    "gbt/model/WidgetData"
], function (declare, lang, Evented, WidgetData) {

    return declare([Evented], {

        /**
        * Creates a WidgetDataManager
        * @param figurePeerNode  - MW/peermodel/PeerNode
        */
        constructor: function (figurePeerNode) {

            // Assertions
            if (!figurePeerNode) {
                throw "Figure PeerNode must be non empty / non null for WidgetDataManager";
            }

            // A Map of <peerNodeIDs, WidgetData]>
            // This map maintains which PeerNode is associated with which WidgetData
            this._WidgetMap = {};

            // Handle to PeerNode for Web Figure
            this._figurePeerNode = figurePeerNode;

            // Handle to a gbt/data/WidgetData of type "Figure"
            this._FigureWidgetData = this._createWidgetData(figurePeerNode);

            // listen for future children under Web Figure
            this._addListenersToPeerNode(figurePeerNode);
        },

        /**
        * Listen to children being added / removed on the peer node
        * */
        _addListenersToPeerNode : function (figurePeerNode) {

            //Listen to the childAdded event on the peerNode
            figurePeerNode.addEventListener("childAdded", lang.hitch(this, "_childAdded"));

            //Listen to the childRemoved event on the peerNode
            figurePeerNode.addEventListener("childRemoved", lang.hitch(this, "_childRemoved"));
        },

         /*
         * Handle child Added event on the peernode
          */
        _childAdded : function (event) {
            var newComponentNode = event.data.child;

            //Create the Widget data of the Component PeerNode
            var childWidgetData = this._createWidgetData(newComponentNode);

            // listen for future adds / removes on the component PeerNode
            this._addListenersToPeerNode(newComponentNode);
        },

        /*
         * Handle child removed event on the peernode
         */
        _childRemoved : function (event) {
            var removeComponentNode = event.data.child;
            this._removeWidgetData(removeComponentNode);
        },


        /**
        * Cleans up this WidgetDataManager
        */
        destroy: function () {
            this._figurePeerNode.destroy();
        },

        /**
         * Gets the WidgetData corresponding to the Figure
         * @return {gbt/model/WidgetData}
         */
        getFigureWindowWidgetData: function () {
            return this._FigureWidgetData;
        },

        /**
         * ------------------------------------------------------------------------------------------------------
         * Private Functions
         * ------------------------------------------------------------------------------------------------------
         */

        /**
        * Handler for when a node is removed
        * @param peerNode: the removed Peer Node
        */
        _removeWidgetData: function (peerNode) {

            // Extract the WidgetData and destroy it
            var peerNodeID = peerNode.getId();

            //get the WidgetData of peerNode to be removed
            var removedWidgetData = this._WidgetMap[peerNodeID];

            //Remove the Children from the FigureWidgetData
            this._FigureWidgetData.removeChild(removedWidgetData);

            removedWidgetData.destroy();
           // remove the entry from the map
            delete this._WidgetMap[peerNodeID];
        },

        _createWidgetData: function (peerNode) {
            // make the widget data
            var widgetData = new WidgetData(peerNode);

            // Store the results in the Map
            var peerNodeID = peerNode.getId();
            this._WidgetMap[peerNodeID] = widgetData;

            // get the parent WidgetData and add it as a child
            var varParentNode = peerNode.getParent();

            //Check to see if peernode has parent then emit WidgetDataAdded to add
            //children under the figure
            if (varParentNode !== undefined) {
                var parentWidgetData =  this._WidgetMap[varParentNode.getId()];

                if (parentWidgetData !== undefined) {
                    // tell the parent it has new children
                    parentWidgetData.addChild(widgetData);
                }
            }
            return widgetData;
        }

    });
});
