/*
 * File: aeroatmprofile210c.c
 *
 * Abstract:
 *
 *   Functions to calculate Non-Standard atmospheric 
 *   condition implemented from MIL-STD-210C.  Given 
 *   geometric altitude in meters, calculate temperature (K), 
 *   pressure (Pa), and density (kg/m^3) using standard
 *   interpolation formula (linear in temperature and
 *   logarithmic in pressure and density).
 *
 *   No extrapolation of values beyond the table range:
 *
 *       IE:  0 <= altitude <= 80000 m
 *
 *   Density/Pressure are calculated using a perfect gas 
 *   relationship depending on lookup table outputs.
 *
 *   Data used are from the 9-JAN-1987 MIL-STD-210C
 *   Global Climatic Data for Developing Military Products 
 *   published by the U.S. Department of Defense, Washington, 
 *   D.C.  The MIL-STC-210C can be obtained from:
 *   
 *     ASSIST Online
 *     http://astimage.daps.dla.mil/online
 *
 *     
 * Copyright 1990-2013 The MathWorks, Inc.
 *
 * Author:  S. Gage      14-AUG-2002
 */

/* MIL-STD-210C atmosphere profile models */

#include <math.h>
#include <stddef.h>
#include "export/include/aeroatmos/aeroatm.hpp"
#include "export/include/aeroatmos/aeroatmosstruct.hpp"
#include "export/include/aeroatmos/aeroatm210c.hpp"
#include "export/include/aeroatmos/aerointerpolate.hpp"

#define NUMPROFILEPTS 41

static double altitude210c[NUMPROFILEPTS] = {  /* in m */
    0.0,  2000.0,  4000.0,  6000.0,  8000.0, 10000.0, 
    12000.0, 14000.0, 16000.0, 18000.0, 20000.0, 
    22000.0, 24000.0, 26000.0, 28000.0, 30000.0, 
    32000.0, 34000.0, 36000.0, 38000.0, 40000.0, 
    42000.0, 44000.0, 46000.0, 48000.0, 50000.0, 
    52000.0, 54000.0, 56000.0, 58000.0, 60000.0, 
    62000.0, 64000.0, 66000.0, 68000.0, 70000.0, 
    72000.0, 74000.0, 76000.0, 78000.0, 80000.0 };


/* Function: InitCalcAtmosEnvelope210c ======================================
 * Abstract:
 *   initialize tables for selected non-standard day profile
 *
 */
void InitCalcAtmosProfile210c( SFcnCache *udata,
                               double *altitude,
                               double *temperature210c,
                               double *density210c,
							   double *minmax)
{

    size_t i;

    minmax[0] = NUMPROFILEPTS - 1;
    minmax[1] = altitude210c[0];
    minmax[2] = altitude210c[NUMPROFILEPTS - 1];

    for ( i = 0; i < NUMPROFILEPTS; i++ )
    {
        altitude[i] = altitude210c[i];
    }

    switch (udata->pvar) {
      case HIGHTEMP:
        switch (udata->palt) {
          case K5:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double ht5k1p[NUMPROFILEPTS] = {  /* in K */ 
                    290.150000,	294.163000,	284.183000,	272.745000,    
                    256.796000,
                    240.858000,	224.930000,	210.558000,	201.411000,    
                    204.873000,
                    212.817000,	216.953000,	220.920000,	224.885000,    
                    228.847000,
                    232.806000,	236.763000,	240.718000,	244.670000,    
                    248.620000,
                    252.567000,	256.512000,	260.454000,	264.394000,    
                    266.150000,
                    266.150000,	262.796000,	258.473000,	253.609000,    
                    246.936000,
                    240.268000,	233.604000,	226.943000,	220.287000,    
                    213.636000,
                    206.988000,	202.695000,	200.156000,	197.619000,    
                    195.084000,
                    192.550000 }; 

                static double htd5k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.215000e+000,	9.506000e-001,	7.773000e-001,	
                    6.343000e-001,	5.209000e-001,
                    4.224000e-001,	3.377000e-001,	2.637000e-001,  
                    1.982000e-001,	1.392000e-001,
                    9.689000e-002,	6.933000e-002,	4.996000e-002,	
                    3.622000e-002,	2.641000e-002,
                    1.937000e-002,	1.428000e-002,	1.058000e-002,	
                    7.885000e-003,	5.902000e-003,
                    4.439000e-003,	3.354000e-003,	2.546000e-003,	
                    1.940000e-003,	1.496000e-003,
                    1.162000e-003,	9.138000e-004,	7.181000e-004,	
                    5.633000e-004,	4.425000e-004,
                    3.454000e-004,	2.678000e-004,	2.061000e-004,	
                    1.574000e-004,	1.193000e-004,
                    8.965000e-005,	6.603000e-005,	4.801000e-005,	
                    3.478000e-005,	2.509000e-005,
                    1.803000e-005 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ht5k1p[i];
                        density210c[i] = htd5k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double ht5k10p[NUMPROFILEPTS] = {  /* in K */ 
                    291.650000,	291.133000,	283.190000,	268.276000,    
                    253.337000,
                    238.407000,	223.496000,	208.575000,	193.674000,    
                    196.813000,
                    204.751000,	212.683000,	217.607000,	221.173000,    
                    224.735000,
                    228.296000,	231.855000,	235.411000,	238.965000,    
                    242.517000,
                    246.066000,	249.613000,	253.159000,	256.701000,    
                    260.150000,
                    260.150000,	260.150000,	260.150000,	254.522000,    
                    246.679000,
                    238.840000,	231.006000,	223.177000,	215.353000,    
                    210.419000,
                    207.489000,	204.560000,	201.634000,	198.709000,    
                    195.786000,
                    192.864000 }; 

                static double htd5k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.209000e+000,	9.571000e-001,	7.761000e-001,	
                    6.399000e-001,	5.220000e-001,
                    4.206000e-001,	3.343000e-001,	2.615000e-001,	
                    2.009000e-001,	1.393000e-001,
                    9.559000e-002,	6.651000e-002,	4.748000e-002,	
                    3.432000e-002,	2.494000e-002,
                    1.821000e-002,	1.337000e-002,	9.868000e-003,	
                    7.315000e-003,	5.447000e-003,
                    4.074000e-003,	3.061000e-003,	2.309000e-003,	
                    1.749000e-003,	1.331000e-003,
                    1.028000e-003,	7.945000e-004,	6.139000e-004,	
                    4.839000e-004,	3.822000e-004,
                    2.996000e-004,	2.330000e-004,	1.796000e-004,	
                    1.372000e-004,	1.025000e-004,
                    7.557000e-005,	5.545000e-005,	4.052000e-005,	
                    2.947000e-005,	2.134000e-005,
                    1.538000e-005 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ht5k10p[i];
                        density210c[i] = htd5k10p[i];
                    }
                  }
                break;
            }break;
          case K10:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double ht10k1p[NUMPROFILEPTS] = {  /* in K */ 
                    299.150000,	289.177000,	279.210000,	269.249000,    
                    259.295000,
                    245.985000,	229.084000,	212.193000,	195.314000,    
                    186.150000,
                    202.781000,	211.443000,	216.197000,	220.948000,    
                    225.696000,
                    230.441000,	235.183000,	239.922000,	244.659000,    
                    249.392000,
                    254.122000,	258.849000,	263.574000,	268.150000,    
                    268.150000,
                    268.150000,	268.150000,	264.940000,	258.077000,    
                    251.217000,
                    244.362000,	237.511000,	230.664000,	223.822000,    
                    216.984000,
                    210.150000,	203.320000,	198.150000,	198.150000,    
                    198.150000,
                    198.150000 }; 

                static double htd10k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.173000e+000,	9.632000e-001,	7.850000e-001,	
                    6.351000e-001,	5.098000e-001,
                    4.110000e-001,	3.315000e-001,	2.630000e-001,	
                    2.048000e-001,	1.500000e-001,
                    9.710000e-002,	6.729000e-002,	4.795000e-002,	
                    3.443000e-002,	2.490000e-002,
                    1.814000e-002,	1.330000e-002,	9.816000e-003,	
                    7.287000e-003,	5.442000e-003,
                    4.087000e-003,	3.087000e-003,	2.343000e-003,	
                    1.789000e-003,	1.392000e-003,
                    1.084000e-003,	8.443000e-004,	6.650000e-004,	
                    5.284000e-004,	4.173000e-004,
                    3.275000e-004,	2.552000e-004,	1.975000e-004,	
                    1.517000e-004,	1.156000e-004,
                    8.736000e-005,	6.541000e-005,	4.811000e-005,	
                    3.438000e-005,	2.457000e-005,
                    1.756000e-005 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ht10k1p[i];
                        density210c[i] = htd10k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double ht10k10p[NUMPROFILEPTS] = {  /* in K */ 
                    300.650000,	289.682000,	278.721000,	267.767000,    
                    256.820000,
                    243.005000,	226.108000,	209.221000,	195.008000,    
                    189.610000,
                    206.663000,	211.757000,	216.114000,	220.468000,    
                    224.819000,
                    229.168000,	233.514000,	237.857000,	242.198000,    
                    246.535000,
                    250.870000,	255.203000,	259.532000,	263.859000,    
                    268.150000,
                    268.150000,	268.150000,	268.150000,	262.555000,    
                    256.677000,
                    250.803000,	244.932000,	239.065000,	233.201000,    
                    227.893000,
                    223.013000,	218.136000,	213.262000,	208.391000,    
                    203.523000,
                    198.658000 }; 

                static double htd10k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.171000e+000,	9.651000e-001,	7.894000e-001,	
                    6.405000e-001,	5.153000e-001,
                    4.152000e-001,	3.340000e-001,	2.642000e-001,	
                    2.024000e-001,	1.457000e-001,
                    9.496000e-002,	6.709000e-002,	4.791000e-002,	
                    3.445000e-002,	2.494000e-002,
                    1.817000e-002,	1.332000e-002,	9.825000e-003,	
                    7.286000e-003,	5.433000e-003,
                    4.073000e-003,	3.069000e-003,	2.324000e-003,	
                    1.768000e-003,	1.352000e-003,
                    1.052000e-003,	8.197000e-004,	6.384000e-004,	
                    5.066000e-004,	4.004000e-004,
                    3.148000e-004,	2.461000e-004,	1.913000e-004,	
                    1.478000e-004,	1.132000e-004,
                    8.610000e-005,	6.507000e-005,	4.887000e-005,	
                    3.647000e-005,	2.703000e-005,
                    1.990000e-005 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ht10k10p[i];
                        density210c[i] = htd10k10p[i];
                    }
                  }
                break;
            }break;
          case K20:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double ht20k1p[NUMPROFILEPTS] = {  /* in K */ 
                    228.150000,	238.650000,	231.113000,	221.103000,     
                    216.150000,
                    217.659000,	219.657000,	221.654000,	223.650000,     
                    232.225000,
                    240.796000,	243.192000,	239.210000,	235.232000,     
                    231.255000,
                    230.150000,	230.150000,	230.150000,	230.150000,     
                    230.150000,
                    230.150000,	230.150000,	230.150000,	230.150000,     
                    232.791000,
                    235.753000,	238.713000,	241.671000,	240.994000,     
                    239.615000,
                    238.237000,	236.860000,	235.484000,	235.150000,     
                    235.150000,
                    235.150000,	233.501000,	231.149000,	228.799000,     
                    226.450000,
                    224.102000 }; 

                static double htd20k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.552000e+000,	1.110000e+000,	8.575000e-001,	
                    6.623000e-001,	4.947000e-001,
                    3.584000e-001,	2.599000e-001,	1.891000e-001,	
                    1.379000e-001,	9.855000e-002,
                    7.126000e-002,	5.336000e-002,	4.092000e-002,	
                    3.124000e-002,	2.375000e-002,
                    1.777000e-002,	1.323000e-002,	9.860000e-003,	
                    7.346000e-003,	5.473000e-003,
                    4.079000e-003,	3.040000e-003,	2.267000e-003,	
                    1.690000e-003,	1.248000e-003,
                    9.242000e-004,	6.870000e-004,	5.126000e-004,	
                    3.890000e-004,	2.957000e-004,
                    2.244000e-004,	1.701000e-004,	1.287000e-004,	
                    9.693000e-005,	7.287000e-005,
                    5.480000e-005,	4.147000e-005,	3.140000e-005,	
                    2.371000e-005,	1.786000e-005,
                    1.341000e-005 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ht20k1p[i];
                        density210c[i] = htd20k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double ht20k10p[NUMPROFILEPTS] = {  /* in K */ 
                    273.150000,	263.128000,	253.105000,	241.094000,    
                    229.090000,
                    228.701000,	234.150000,	234.150000,	234.150000,     
                    234.150000,
                    234.150000,	234.150000,	234.150000,	234.150000,     
                    237.071000,
                    240.051000,	243.030000,	247.720000,	252.480000,     
                    257.932000,
                    264.270000,	270.605000,	276.935000,	283.262000,     
                    287.150000,
                    287.150000,	287.150000,	284.317000,	279.587000,     
                    274.859000,
                    270.135000,	265.414000,	260.696000,	253.933000,     
                    240.974000,
                    228.023000,	215.079000,	202.144000,	189.217000,     
                    176.298000,
                    163.387000 }; 

                static double htd20k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.293000e+000,	1.040000e+000,	8.296000e-001,	
                    6.603000e-001,	5.196000e-001,
                    3.847000e-001,	2.802000e-001,	2.094000e-001,	
                    1.565000e-001,	1.170000e-001,
                    8.748000e-002,	6.542000e-001,	4.893000e-002,	
                    3.660000e-002,	2.709000e-002,
                    2.013000e-002,	1.501000e-002,	1.117000e-002,	
                    8.363000e-003,	6.277000e-003,
                    4.728000e-003,	3.585000e-003,	2.737000e-003,	
                    2.102000e-003,	1.638000e-003,
                    1.295000e-003,	1.024000e-003,	8.174000e-004,	
                    6.547000e-004,	5.224000e-004,
                    4.153000e-004,	3.288000e-004,	2.593000e-004,	
                    2.053000e-004,	1.649000e-004,
                    1.309000e-004,	1.026000e-004,	7.919000e-005,	
                    6.009000e-005,	4.472000e-005,
                    3.255000e-005 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ht20k10p[i];
                        density210c[i] = htd20k10p[i];
                    }
                  }
                break;
            } break;
          case K30:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double ht30k1p[NUMPROFILEPTS] = {  /* in K */ 
                    279.150000,	275.124000,	262.106000,	249.097000,     
                    236.095000,
                    223.162000,	226.358000,	227.751000,	228.549000,     
                    229.346000,
                    230.143000,	235.087000,	240.063000,	245.035000,     
                    250.005000,
                    254.971000,	259.386000,	263.156000,	266.923000,     
                    270.688000,
                    274.451000,	278.211000,	281.969000,	285.724000,     
                    286.150000,
                    286.150000,	286.150000,	282.406000,	276.100000,     
                    269.799000,
                    263.501000,	257.207000,	250.917000,	243.303000,     
                    231.524000,
                    219.752000,	207.988000,	196.231000,	184.481000,     
                    172.739000,
                    161.004000 }; 

                static double htd30k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.262000e+000,	1.000000e+000,	8.137000e-001,	
                    6.552000e-001,	5.216000e-001,
                    4.098000e-001,	2.982000e-001,	2.195000e-001,	
                    1.622000e-001,	1.201000e-001,
                    8.899000e-002,	6.502000e-002,	4.782000e-002,	
                    3.540000e-002,	2.637000e-002,
                    1.976000e-002,	1.492000e-002,	1.135000e-002,	
                    8.668000e-003,	6.644000e-003,
                    5.113000e-003,	3.949000e-003,	3.061000e-003,	
                    2.381000e-003,	1.878000e-003,
                    1.483000e-003,	1.172000e-003,	9.379000e-004,	
                    7.538000e-004,	6.029000e-004,
                    4.797000e-004,	3.796000e-004,	2.988000e-004,	
                    2.350000e-004,	1.861000e-004,
                    1.457000e-004,	1.125000e-004,	8.565000e-005,	
                    6.410000e-005,	4.707000e-005,
                    3.383000e-005 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ht30k1p[i];
                        density210c[i] = htd30k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double ht30k10p[NUMPROFILEPTS] = {  /* in K */ 
                    285.150000,	274.135000,	263.113000,	245.112000,     
                    227.140000,
                    223.652000,	227.645000,	228.650000,	228.650000,     
                    228.650000,
                    228.650000,	230.611000,	232.599000,	234.587000,     
                    239.651000,
                    244.813000,	249.971000,	255.126000,	260.278000,     
                    265.427000,
                    270.572000,	275.714000,	280.853000,	285.989000,     
                    286.650000,
                    286.650000,	286.650000,	283.813000,	278.890000,     
                    273.970000,
                    269.054000,	264.140000,	256.228000,	245.235000,     
                    234.248000,
                    223.269000,	212.296000,	201.331000,	190.372000,     
                    179.420000,
                    168.474000 }; 

                static double htd30k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.235000e+000,	1.006000e+000,	8.130000e-001,	
                    6.669000e-001,	5.388000e-001,
                    4.037000e-001,	2.932000e-001,	2.166000e-001,	
                    1.608000e-001,	1.194000e-001,
                    8.868000e-002,	6.539000e-002,	4.834000e-002,	
                    3.584000e-002,	2.635000e-002,
                    1.949000e-002,	1.451000e-002,	1.087000e-002,	
                    8.200000e-003,	6.216000e-003,
                    4.738000e-003,	3.630000e-003,	2.796000e-003,	
                    2.164000e-003,	1.706000e-003,
                    1.349000e-003,	1.066000e-003,	8.511000e-004,	
                    6.819000e-004,	5.443000e-004,
                    4.327000e-004,	3.426000e-004,	2.731000e-004,	
                    2.184000e-004,	1.728000e-004,
                    1.353000e-004,	1.046000e-004,	7.985000e-005,	
                    6.002000e-005,	4.436000e-005,
                    3.218000e-005 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ht30k10p[i];
                        density210c[i] = htd30k10p[i];
                    }
                  }
                break;
            } break;
          case K40:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double ht40k1p[NUMPROFILEPTS] = {  /* in K */ 
                    287.150000,	273.131000,	259.121000,	245.120000,     
                    231.128000,
                    225.651000,	228.646000,	228.650000,	228.650000,     
                    228.650000,
                    229.814000,	232.203000,	234.589000,	236.975000,     
                    241.209000,
                    246.172000,	251.132000,	256.089000,	262.535000,     
                    269.664000,
                    276.788000,	283.908000,	284.650000,	284.650000,     
                    284.650000,
                    282.861000,	279.904000,	276.948000,	273.994000,     
                    271.042000,
                    268.092000,	265.144000,	258.034000,	246.648000,     
                    235.270000,
                    223.898000,	212.534000,	201.176000,	189.826000,     
                    178.483000,
                    167.146000 }; 

                static double htd40k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.227000e+000,	1.010000e+000,	8.239000e-001,	
                    6.641000e-001,	5.287000e-001,
                    4.005000e-001,	2.927000e-001,	2.172000e-001,	
                    1.613000e-001,	1.197000e-001,
                    8.852000e-002,	6.527000e-002,	4.829000e-002,	
                    3.584000e-002,	2.650000e-002,
                    1.966000e-002,	1.467000e-002,	1.101000e-002,	
                    8.275000e-003,	6.247000e-003,
                    4.752000e-003,	3.641000e-003,	2.864000e-003,	
                    2.259000e-003,	1.783000e-003,
                    1.415000e-003,	1.125000e-003,	8.935000e-004,	
                    7.075000e-004,	5.588000e-004,
                    4.403000e-004,	3.461000e-004,	2.754000e-004,	
                    2.209000e-004,	1.753000e-004,
                    1.376000e-004,	1.066000e-004,	8.155000e-005,	
                    6.138000e-005,	4.541000e-005,
                    3.295000e-005 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ht40k1p[i];
                        density210c[i] = htd40k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double ht40k10p[NUMPROFILEPTS] = {  /* in K */ 
                    283.150000,	271.134000,	259.125000,	247.124000,     
                    235.131000,
                    226.551000,	228.148000,	228.150000,	228.150000,     
                    228.150000,
                    228.150000,	230.503000,	232.889000,	235.275000,     
                    238.421000,
                    244.376000,	250.328000,	256.276000,	262.221000,     
                    268.162000,
                    274.099000,	278.841000,	281.806000,	284.769000,     
                    286.650000,
                    286.650000,	285.454000,	281.908000,	278.363000,     
                    274.821000,
                    271.281000,	267.743000,	264.207000,	252.562000,     
                    240.791000,
                    229.027000,	217.271000,	205.522000,	193.780000,     
                    182.046000,
                    170.319000 }; 

                static double htd40k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.244000e+000,	1.015000e+000,	8.209000e-001,	
                    6.571000e-001,	5.202000e-001,
                    4.011000e-001,	2.950000e-001,	2.188000e-001,	
                    1.623000e-001,	1.204000e-001,
                    8.941000e-002,	6.579000e-002,	4.856000e-002,	
                    3.596000e-002,	2.664000e-002,
                    1.962000e-002,	1.456000e-002,	1.089000e-002,	
                    8.197000e-003,	6.210000e-003,
                    4.735000e-003,	3.646000e-003,	2.835000e-003,	
                    2.211000e-003,	1.735000e-003,
                    1.371000e-003,	1.088000e-003,	8.697000e-004,	
                    6.927000e-004,	5.502000e-004,
                    4.358000e-004,	3.442000e-004,	2.710000e-004,	
                    2.187000e-004,	1.748000e-004,
                    1.382000e-004,	1.079000e-004,	8.313000e-005,	
                    6.307000e-005,	4.704000e-005,
                    3.441000e-005 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ht40k10p[i];
                        density210c[i] = htd40k10p[i];
                    }
                  }
                break;
            } break;
        } break;
      case LOWTEMP:
        switch (udata->palt) {
          case K5:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double lt5k1p[NUMPROFILEPTS] = {  /* in K */ 
                    220.150000,	234.142000,	224.860000,	216.141000,    
                    214.141000,
                    211.633000,	207.637000,	203.643000,	199.652000,     
                    199.650000,
                    199.650000,	199.650000,	207.129000,	214.688000,     
                    222.243000,
                    229.792000,	237.337000,	244.877000,	250.612000,     
                    255.567000,
                    260.518000,	265.467000,	269.650000,	269.650000,     
                    268.438000,
                    265.279000,	262.122000,	258.967000,	255.814000,     
                    252.663000,
                    249.317000,	241.646000,	233.979000,	230.150000,     
                    230.150000,
                    230.150000,	228.507000,	226.155000,	223.805000,     
                    221.456000,
                    219.109000 }; 

                static double ltd5k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.607000e+000,	1.124000e+000,	8.698000e-001,	
                    6.628000e-001,	4.869000e-001,
                    3.575000e-001,	2.631000e-001,	1.925000e-001,	
                    1.400000e-001,	9.958000e-002,
                    7.080000e-002,	5.035000e-002,	3.474000e-002,	
                    2.428000e-002,	1.719000e-002,
                    1.231000e-002,	8.916000e-003,	6.523000e-003,	
                    4.850000e-003,	3.639000e-003,
                    2.747000e-003,	2.084000e-003,	1.594000e-003,	
                    1.241000e-003,	9.706000e-004,
                    7.628000e-004,	5.978000e-004,	4.672000e-004,	
                    3.641000e-004,	2.829000e-004,
                    2.193000e-004,	1.721000e-004,	1.340000e-004,	
                    1.019000e-004,	7.615000e-005,
                    5.691000e-005,	4.281000e-005,	3.222000e-005,	
                    2.418000e-005,	1.810000e-005,
                    1.350000e-005 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = lt5k1p[i];
                        density210c[i] = ltd5k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double lt5k10p[NUMPROFILEPTS] = {  /* in K */ 
                    229.650000,	236.633000,	228.621000,	219.105000,     
                    209.150000,
                    209.150000,	209.137000,	205.143000,	202.150000,
                    202.150000,
                    202.150000,	206.061000,	214.023000,	221.980000,     
                    229.932000,
                    237.879000,	245.821000,	252.753000,	255.727000,    
                    258.700000,
                    261.671000,	263.889000,	262.306000,	260.725000,    
                    259.144000,
                    257.565000,	255.986000,	254.409000,	252.832000,    
                    251.256000,
                    248.101000,	241.217000,	234.336000,	229.150000,    
                    229.150000,
                    229.150000,	227.781000,	225.821000,	223.862000,    
                    221.905000,
                    219.949000 }; 

                static double ltd5k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.541000e+000,	1.117000e+000,	8.620000e-001,	
                    6.628000e-001,	5.045000e-001,
                    3.640000e-001,	2.626000e-001,	1.926000e-001,	
                    1.397000e-001,	9.979000e-002,
                    7.126000e-002,	5.000000e-002,	3.482000e-002,	
                    2.458000e-002,	1.757000e-002,
                    1.270000e-002,	9.289000e-003,	6.886000e-003,	
                    5.214000e-003,	3.961000e-003,
                    3.019000e-003,	2.315000e-003,	1.801000e-003,	
                    1.400000e-003,	1.086000e-003,
                    8.420000e-004,	6.516000e-004,	5.035000e-004,	
                    3.885000e-004,	2.993000e-004,
                    2.317000e-004,	1.810000e-004,	1.405000e-004,	
                    1.074000e-004,	8.021000e-005,
                    5.987000e-005,	4.493000e-005,	3.374000e-005,	
                    2.527000e-005,	1.888000e-005,
                    1.408000e-005 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = lt5k10p[i];
                        density210c[i] = ltd5k10p[i];
                    }
                  }
                break;
            } break;
          case K10:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double lt10k1p[NUMPROFILEPTS] = {  /* in K */ 
                    237.150000,	259.625000,	247.607000,	233.074000,    
                    216.070000,
                    199.159000,	201.150000,	201.150000,	201.150000,     
                    201.150000,
                    203.319000,	207.701000,	212.080000,	216.456000,
                    220.830000,
                    225.201000,	229.569000,	233.934000,	238.297000,
                    240.470000,
                    241.658000,	242.846000,	244.033000,	245.219000,  
                    246.404000,
                    247.589000,	248.773000,	249.956000,	251.139000,
                    248.299000,
                    242.394000,	241.771000,	242.557000,	243.343000,
                    244.128000,
                    244.913000,	244.055000,	242.487000,	240.920000,
                    239.354000,
                    237.789000, }; 

                static double ltd10k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.497000e+000,	1.043000e+000,	8.349000e-001,	
                    6.678000e-001,	5.312000e-001,
                    4.147000e-001,	2.922000e-001,	2.079000e-001,	
                    1.482000e-001,	1.056000e-001,
                    7.455000e-002,	5.241000e-002,	3.712000e-002,	
                    2.648000e-002,	1.903000e-002,
                    1.376000e-002,	1.001000e-002,	7.338000e-003,	
                    5.407000e-003,	4.040000e-003,
                    3.036000e-003,	2.285000e-003,	1.723000e-003,	
                    1.301000e-003,	9.839000e-004,
                    7.452000e-004,	5.652000e-004,	4.294000e-004,	
                    3.266000e-004,	2.526000e-004,
                    1.967000e-004,	1.493000e-004,	1.127000e-004,	
                    8.529000e-005,	6.456000e-005,
                    4.892000e-005,	3.734000e-005,	2.854000e-005,	
                    2.177000e-005,	1.659000e-005,
                    1.262000e-005 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = lt10k1p[i];
                        density210c[i] = ltd10k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double lt10k10p[NUMPROFILEPTS] = {  /* in K */ 
                    246.150000,	258.124000,	246.105000,	232.681000,    
                    217.876000,
                    203.150000,	203.137000,	199.542000,	195.950000,    
                    192.360000,
                    188.773000,	185.187000,	188.105000,	191.089000,    
                    194.071000,
                    197.051000,	202.882000,	212.010000,	221.132000,   
                    230.249000,
                    239.361000,	248.466000,	257.566000,	266.660000,    
                    275.749000,
                    279.150000,	279.150000,	279.150000,	279.150000,    
                    267.802000,
                    254.024000,	240.253000,	226.491000,	215.516000,    
                    212.964000,
                    210.413000,	207.863000,	205.316000,	203.150000,    
                    203.150000,
                    203.150000 }; 

                static double ltd10k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.438000e+000,	1.048000e+000,	8.385000e-001,	
                    6.667000e-001,	5.256000e-001,
                    4.074000e-001,	2.912000e-001,	2.112000e-001,	
                    1.523000e-001,	1.092000e-001,
                    7.790000e-002,	5.518000e-002,	3.773000e-002,	
                    2.595000e-002,	1.796000e-002,
                    1.250000e-002,	8.638000e-003,	5.961000e-003,	
                    4.180000e-003,	2.973000e-003,
                    2.144000e-003,	1.565000e-003,	1.156000e-003,	
                    8.630000e-004,	6.507000e-004,
                    5.045000e-004,	3.962000e-004,	3.113000e-004,	
                    2.445000e-004,	1.995000e-004,
                    1.625000e-004,	1.309000e-004,	1.041000e-004,	
                    8.065000e-005,	5.967000e-005,
                    4.400000e-005,	3.233000e-005,	2.367000e-005,	
                    1.723000e-005,	1.241000e-005,
                    8.925000e-006 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = lt10k10p[i];
                        density210c[i] = ltd10k10p[i];
                    }
                  }
                break;
            } break;
          case K20:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double lt20k1p[NUMPROFILEPTS] = {  /* in K */ 
                    236.650000,	250.023000,	237.604000,	225.192000,    
                    212.788000,
                    206.650000,	206.631000,	201.639000,	196.650000,    
                    191.664000,
                    186.682000,	189.619000,	192.605000,	198.920000,    
                    205.282000,
                    211.640000,	217.994000,	224.344000,	230.690000,    
                    237.032000,
                    243.370000,	249.705000,	256.035000,	262.362000,    
                    264.650000,
                    264.650000,	264.650000,	264.650000,	263.152000,    
                    258.622000,
                    254.094000,	245.367000,	235.537000,	227.468000,    
                    224.719000,
                    221.971000,	219.226000,	216.482000,	213.740000,    
                    211.000000,
                    208.261000 }; 

                static double ltd20k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.496000e+000,	1.072000e+000,	8.525000e-001,	
                    6.693000e-001,	5.183000e-001,
                    3.844000e-001,	2.763000e-001,	2.027000e-001,	
                    1.476000e-001,	1.066000e-001,
                    7.637000e-002,	5.237000e-002,	3.612000e-002,	
                    2.471000e-002,	1.711000e-002,
                    1.198000e-002,	8.484000e-003,	6.067000e-003,	
                    4.380000e-003,	3.191000e-003,
                    2.345000e-003,	1.737000e-003,	1.297000e-003,	
                    9.754000e-004,	7.490000e-004,
                    5.805000e-004,	4.499000e-004,	3.488000e-004,	
                    2.719000e-004,	2.138000e-004,
                    1.673000e-004,	1.325000e-004,	1.043000e-004,	
                    8.082000e-005,	6.080000e-005,
                    4.559000e-005,	3.407000e-005,	2.537000e-005,	
                    1.883000e-005,	1.392000e-005,
                    1.025000e-005 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = lt20k1p[i];
                        density210c[i] = ltd20k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double lt20k10p[NUMPROFILEPTS] = {  /* in K */ 
                    240.150000,	254.130000,	245.095000,	230.482000,    
                    215.877000,
                    207.136000,	204.139000,	201.144000,	198.150000,    
                    195.159000,
                    192.169000,	192.150000,	192.150000,	196.068000,    
                    200.045000,
                    204.019000,	207.990000,	211.959000,	219.700000,    
                    227.628000,
                    235.551000,	243.468000,	251.381000,	259.289000,    
                    264.150000,
                    264.150000,	264.150000,	264.150000,	262.977000,    
                    259.432000,
                    255.889000,	247.055000,	236.832000,	228.858000,    
                    227.679000,
                    226.502000,	223.951000,	220.815000,	217.681000,    
                    214.549000,
                    211.420000, }; 

                static double ltd20k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.472000e+000,	1.054000e+000,	8.312000e-001,	
                    6.629000e-001,	5.210000e-001,
                    3.922000e-001,	2.855000e-001,	2.069000e-001,	
                    1.493000e-001,	1.072000e-001,
                    7.661000e-002,	5.377000e-002,	3.774000e-002,	
                    2.606000e-002,	1.812000e-002,
                    1.270000e-002,	8.963000e-003,	6.368000e-003,	
                    4.488000e-003,	3.200000e-003,
                    2.308000e-003,	1.684000e-003,	1.241000e-003,	
                    9.237000e-004,	7.012000e-004,
                    5.432000e-004,	4.208000e-004,	3.261000e-004,	
                    2.538000e-004,	1.988000e-004,
                    1.552000e-004,	1.231000e-004,	9.735000e-005,	
                    7.543000e-005,	5.651000e-005,
                    4.228000e-005,	3.177000e-005,	2.384000e-005,	
                    1.782000e-005,	1.327000e-005,
                    9.842000e-006 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = lt20k10p[i];
                        density210c[i] = ltd20k10p[i];
                    }
                  }
                break;
            } break;
          case K30:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double lt30k1p[NUMPROFILEPTS] = {  /* in K */ 
                    241.150000,	241.547000,	237.242000,	229.645000,    
                    222.053000,
                    220.150000,	220.150000,	219.657000,	218.661000,    
                    217.665000,
                    216.670000,	208.851000,	200.899000,	195.448000,    
                    192.867000,
                    190.288000,	190.150000,	190.150000,	193.829000,    
                    197.787000,
                    205.328000,	218.177000,	231.018000,	243.851000,    
                    256.676000,
                    258.650000,	255.536000,	251.794000,	248.862000,    
                    247.878000,
                    246.895000,	245.913000,	244.932000,	243.951000,    
                    242.970000,
                    241.990000,	238.583000,	233.886000,	229.192000,    
                    224.500000,
                    219.812000 }; 

                static double ltd30k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.462000e+000,	1.101000e+000,	8.431000e-001,	
                    6.500000e-001,	4.969000e-001,
                    3.678000e-001,	2.698000e-001,	1.984000e-001,	
                    1.461000e-001,	1.074000e-001,
                    7.893000e-002,	5.949000e-002,	4.440000e-002,	
                    3.236000e-002,	2.312000e-002,
                    1.645000e-002,	1.153000e-002,	8.077000e-003,	
                    5.569000e-003,	3.863000e-003,
                    2.656000e-003,	1.817000e-003,	1.270000e-003,	
                    9.061000e-004,	6.575000e-004,
                    5.028000e-004,	3.918000e-004,	3.050000e-004,	
                    2.358000e-004,	1.806000e-004,
                    1.382000e-004,	1.057000e-004,	8.075000e-005,	
                    6.163000e-005,	4.699000e-005,
                    3.580000e-005,	2.750000e-005,	2.113000e-005,	
                    1.616000e-005,	1.229000e-005,
                    9.296000e-006 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = lt30k1p[i];
                        density210c[i] = ltd30k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double lt30k10p[NUMPROFILEPTS] = {  /* in K */ 
                    253.150000,	254.374000,	241.356000,	228.347000,    
                    215.345000,
                    212.150000,	212.150000,	210.648000,	207.655000,     
                    204.664000,
                    201.675000,	198.688000,	195.702000,	194.150000,     
                    194.150000,
                    194.150000,	197.074000,	203.423000,	209.768000,     
                    216.109000,
                    222.446000,	228.779000,	235.108000,	241.433000,     
                    247.754000,
                    254.071000,	258.150000,	258.150000,	256.356000,     
                    250.843000,
                    245.332000,	239.825000,	237.756000,	238.935000,     
                    240.113000,
                    241.290000,	238.657000,	234.346000,	230.038000,     
                    225.733000,
                    221.430000 }; 

                static double ltd30k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.393000e+000,	1.061000e+000,	8.484000e-001,	
                    6.702000e-001,	5.222000e-001,
                    3.844000e-001,	2.786000e-001,	2.033000e-001,	
                    1.489000e-001,	1.085000e-001,
                    7.882000e-002,	5.695000e-002,	4.096000e-002,	
                    2.911000e-002,	2.052000e-002,
                    1.446000e-002,	1.006000e-002,	6.949000e-003,	
                    4.855000e-003,	3.429000e-003,
                    2.447000e-003,	1.763000e-003,	1.282000e-003,	
                    9.403000e-004,	6.954000e-004,
                    5.182000e-004,	3.923000e-004,	3.022000e-004,	
                    2.344000e-004,	1.837000e-004,
                    1.432000e-004,	1.111000e-004,	8.451000e-005,	
                    6.346000e-005,	4.772000e-005,
                    3.595000e-005,	2.751000e-005,	2.111000e-005,	
                    1.612000e-005,	1.225000e-005,
                    9.264000e-006 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = lt30k10p[i];
                        density210c[i] = ltd30k10p[i];
                    }
                  }
                break;
            } break;
          case K40:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double lt40k1p[NUMPROFILEPTS] = {  /* in K */ 
                    253.150000,	255.874000,	242.856000,	229.847000,     
                    216.845000,
                    215.157000,	217.155000,	216.648000,	213.655000,     
                    210.664000,
                    207.675000,	204.688000,	201.702000,	200.150000,     
                    200.150000,
                    200.150000,	200.150000,	200.150000,	200.150000,     
                    200.150000,
                    201.835000,	205.793000,	212.087000,	219.599000,     
                    227.105000,
                    234.607000,	242.104000,	249.596000,	252.792000,     
                    247.082000,
                    241.374000,	242.467000,	245.415000,	248.362000,     
                    251.307000,
                    254.249000,	252.157000,	247.846000,	243.538000,     
                    239.233000,
                    234.930000 }; 

                static double ltd40k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.396000e+000,	1.057000e+000,	8.470000e-001,	
                    6.701000e-001,	5.230000e-001,
                    3.834000e-001,	2.770000e-001,	2.029000e-001,	
                    1.499000e-001,	1.102000e-001,
                    8.079000e-002,	5.893000e-002,	4.280000e-002,	
                    3.073000e-002,	2.189000e-002,
                    1.559000e-002,	1.111000e-002,	7.921000e-003,	
                    5.647000e-003,	4.026000e-003,
                    2.849000e-003,	2.005000e-003,	1.407000e-003,	
                    9.941000e-004,	7.105000e-004,
                    5.135000e-004,	3.750000e-004,	2.766000e-004,	
                    2.092000e-004,	1.635000e-004,
                    1.271000e-004,	9.578000e-005,	7.186000e-005,	
                    5.410000e-005,	4.088000e-005,
                    3.099000e-005,	2.400000e-005,	1.868000e-005,	
                    1.448000e-005,	1.117000e-005,
                    8.587000e-006 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = lt40k1p[i];
                        density210c[i] = ltd40k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double lt40k10p[NUMPROFILEPTS] = {  /* in K */ 
                    253.150000,	255.874000,	242.856000,	229.847000,     
                    216.845000,
                    214.404000,	215.403000,	212.647000,	208.656000,     
                    204.668000,
                    200.683000,	199.663000,	198.667000,	197.673000,     
                    198.233000,
                    198.829000,	199.424000,	200.019000,	200.614000,     
                    205.305000,
                    210.256000,	215.204000,	220.148000,	226.409000,     
                    234.705000,
                    242.997000,	246.528000,	249.288000,	250.189000,     
                    247.235000,
                    244.283000,	241.333000,	242.160000,	244.125000,     
                    246.088000,
                    248.050000,	245.657000,	241.346000,	237.038000,     
                    232.733000,
                    228.430000, }; 

                static double ltd40k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.396000e+000,	1.057000e+000,	8.470000e-001,	
                    6.701000e-001,	5.230000e-001,
                    3.846000e-001,	2.786000e-001,	2.054000e-001,	
                    1.514000e-001,	1.110000e-001,
                    8.094000e-002,	5.791000e-002,	4.137000e-002,	
                    2.950000e-002,	2.087000e-002,
                    1.478000e-002,	1.048000e-002,	7.448000e-003,	
                    5.295000e-003,	3.706000e-003,
                    2.613000e-003,	1.857000e-003,	1.331000e-003,	
                    9.562000e-004,	6.883000e-004,
                    5.012000e-004,	3.753000e-004,	2.828000e-004,	
                    2.153000e-004,	1.662000e-004,
                    1.280000e-004,	9.826000e-005,	7.413000e-005,	
                    5.580000e-005,	4.210000e-005,
                    3.184000e-005,	2.453000e-005,	1.896000e-005,	
                    1.460000e-005,	1.118000e-005,
                    8.530000e-006 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = lt40k10p[i];
                        density210c[i] = ltd40k10p[i];
                    }
                  }
                break;
            } break;
        } break;
      case HIGHDENSITY:
        switch (udata->palt){
          case K5:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double hd5k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.431000e+000,	1.113000e+000,	8.774000e-001,	
                    6.774000e-001,	5.055000e-001,
                    3.701000e-001,	2.714000e-001,	1.993000e-001,	
                    1.469000e-001,	1.086000e-001,
                    8.030000e-002,	5.937000e-002,	4.390000e-002,	
                    3.247000e-002,	2.402000e-002,
                    1.777000e-002,	1.295000e-002,	9.354000e-003,	
                    6.829000e-003,	5.038000e-003,
                    3.753000e-003,	2.854000e-003,	2.215000e-003,	
                    1.718000e-003,	1.334000e-003,
                    1.050000e-003,	8.259000e-004,	6.463000e-004,	
                    5.034000e-004,	3.902000e-004,
                    3.010000e-004,	2.289000e-004,	1.732000e-004, 
                    1.308000e-004,	9.855000e-005,
                    7.407000e-005,	5.596000e-005,	4.221000e-005,	
                    3.165000e-006,	2.359000e-005,
                    1.746000e-005 }; 

                static double hdt5k1p[NUMPROFILEPTS] = {  /* in K */ 
                    251.150000,	245.142000,	234.117000,	224.643000,     
                    221.652000,
                    222.650000,	223.649000,	224.646000,	225.150000,     
                    225.150000,
                    225.150000,	225.150000,	225.150000,	225.150000,     
                    225.150000,
                    225.150000,	228.812000,	236.941000,	245.065000,     
                    253.184000,
                    261.298000,	266.150000,	266.150000,	266.150000,     
                    266.150000,
                    261.917000,	256.988000,	252.062000,	247.139000,     
                    242.219000,
                    237.302000,	234.645000,	232.681000,	230.718000,     
                    228.756000,
                    226.795000,	222.996000,	218.297000,	213.600000,     
                    208.906000,
                    204.215000 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = hdt5k1p[i];
                        density210c[i] = hd5k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double hd5k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.509000e+000,	1.104000e+000,	8.739000e-001,	
                    6.833000e-001,	5.270000e-001,
                    3.772000e-001,	2.712000e-001,	1.977000e-001,	
                    1.454000e-001,	1.067000e-001,
                    7.815000e-002,	5.711000e-002,	4.164000e-002,	
                    3.030000e-002,	2.199000e-002,
                    1.535000e-002,	1.084000e-002,	7.752000e-003,	
                    5.608000e-003,	4.101000e-003,
                    3.029000e-003,	2.270000e-003,	1.751000e-003,	
                    1.354000e-003,	1.052000e-003,
                    8.154000e-004,	6.307000e-004,	4.868000e-004,	
                    3.740000e-004,	2.855000e-004,
                    2.180000e-004,	1.665000e-004,	1.271000e-004,	
                    9.715000e-005,	7.423000e-005,
                    5.672000e-005,	4.391000e-005,	3.405000e-005,	
                    2.626000e-005,	2.014000e-005,
                    1.536000e-005 }; 

                static double hdt5k10p[NUMPROFILEPTS] = {  /* in K */ 
                    238.150000,	247.129000,	235.116000,	223.110000,     
                    211.159000,
                    213.957000,	216.753000,	217.352000,	215.756000,     
                    214.161000,
                    212.567000,	210.975000,	209.383000,	207.792000,     
                    206.202000,
                    213.832000,	221.771000,	229.704000,	237.634000,     
                    245.558000,
                    253.477000,	260.150000,	260.150000,	259.310000,     
                    257.138000,
                    254.967000,	252.797000,	250.629000,	249.150000,     
                    249.150000,
                    249.150000,	249.150000,	249.150000,	249.150000,     
                    249.150000,
                    249.150000,	245.663000,	240.570000,	235.480000,     
                    230.393000,
                    225.309000, }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = hdt5k10p[i];
                        density210c[i] = hd5k10p[i];
                    }
                  }
                break;
            } break;
          case K10:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double hd10k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.203000e+000,	9.786000e-001,	8.035000e-001,	
                    6.668000e-001,	5.331000e-001,
                    4.342000e-001,	3.442000e-001,	2.491000e-001,	
                    1.797000e-001,	1.285000e-001,
                    9.144000e-002,	6.532000e-002,	4.686000e-002,	
                    3.376000e-002,	2.430000e-002,
                    1.751000e-002,	1.271000e-002,	9.294000e-003,	
                    6.843000e-003,	5.072000e-003,
                    3.784000e-003,	2.840000e-003,	2.176000e-003,	
                    1.673000e-003,	1.287000e-003,
                    9.901000e-004,	7.695000e-004,	5.980000e-004,	
                    4.630000e-004,	3.571000e-004,
                    2.744000e-004,	2.100000e-004,	1.601000e-004,	
                    1.215000e-004,	9.182000e-005,
                    6.907000e-005,	5.125000e-005,	3.775000e-005,	
                    2.781000e-005,	2.081000e-005,
                    1.548000e-005 }; 

                static double hdt10k1p[NUMPROFILEPTS] = {  /* in K */ 
                    295.150000,	287.567000,	274.792000,	260.756000,     
                    242.815000,
                    224.884000,	206.965000,	205.673000,	204.678000,     
                    205.360000,
                    207.942000,	210.522000,	213.101000,	215.678000,     
                    219.356000,
                    224.503000,	229.648000,	234.789000,	239.926000,     
                    245.062000,
                    250.192000,	255.321000,	256.150000,	256.150000,     
                    256.150000,
                    256.150000,	253.254000,	249.520000,	245.789000,     
                    242.060000,
                    238.334000,	234.609000,	230.888000,	227.168000,     
                    223.451000,
                    219.736000,	218.150000,	218.150000,	218.103000,     
                    214.202000,
                    210.304000 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = hdt10k1p[i];
                        density210c[i] = hd10k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double hd10k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.228000e+000,	9.948000e-001,	8.005000e-001,	
                    6.596000e-001,	5.365000e-001,
                    4.298000e-001,	3.384000e-001,	2.433000e-001,	
                    1.747000e-001,	1.243000e-001,
                    8.878000e-002,	6.366000e-002,	4.583000e-002,	
                    3.311000e-002,	2.402000e-002,
                    1.748000e-002,	1.277000e-002,	9.366000e-003,	
                    6.891000e-003,	5.027000e-003,
                    3.690000e-003,	2.730000e-003,	2.034000e-003,	
                    1.527000e-003,	1.154000e-003,
                    8.949000e-004,	6.938000e-004,	5.456000e-004,	
                    4.274000e-004,	3.334000e-004,
                    2.591000e-004,	2.004000e-004,	1.543000e-004,	
                    1.183000e-004,	9.031000e-005,
                    6.853000e-005,	5.150000e-005,	3.856000e-005,	
                    2.876000e-005,	2.138000e-005,
                    1.582000e-005 }; 

                static double hdt10k10p[NUMPROFILEPTS] = {  /* in K */ 
                    289.150000,	281.160000,	273.175000,	256.245000,     
                    239.298000,
                    222.361000,	205.436000,	205.150000,	205.150000,     
                    207.469000,
                    209.853000,	212.235000,	214.615000,	216.994000,     
                    219.372000,
                    221.748000,	224.122000,	226.495000,	228.867000,     
                    234.368000,
                    240.289000,	246.207000,	252.121000,	258.032000,     
                    263.650000,
                    263.650000,	263.583000,	259.260000,	254.939000,     
                    250.621000,
                    246.306000,	241.993000,	237.683000,	233.276000,     
                    229.071000,
                    224.890000,	221.959000,	219.029000,	216.102000,     
                    213.176000,
                    210.252000 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = hdt10k10p[i];
                        density210c[i] = hd10k10p[i];
                    }
                  }
                break;
            } break;
          case K20:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double hd20k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.169000e+000,	9.609000e-001,	7.843000e-001,	
                    6.356000e-001,	5.110000e-001,
                    4.160000e-001,	3.342000e-001,	2.643000e-001,	
                    2.053000e-001,	1.475000e-001,
                    9.994000e-002,	6.891000e-002,	4.929000e-002,	
                    3.554000e-002,	2.578000e-002,
                    1.880000e-002,	1.380000e-002,	1.017000e-002,	
                    7.548000e-003,	5.603000e-003,
                    4.159000e-003,	3.112000e-003,	2.347000e-003,	
                    1.795000e-003,	1.401000e-003,
                    1.094000e-003,	8.710000e-004,	6.899000e-004,	
                    5.437000e-004,	4.261000e-004,
                    3.321000e-004,	2.572000e-004,	1.980000e-004,	
                    1.514000e-004,	1.150000e-004,
                    8.673000e-005,	6.489000e-005,	4.815000e-005,	
                    3.542000e-005,	2.582000e-005,
                    1.805000e-005 }; 

                static double hdt20k1p[NUMPROFILEPTS] = {  /* in K */ 
                    301.150000,	291.179000,	281.214000,	271.256000,     
                    261.304000,
                    245.483000,	229.580000,	213.687000,	197.803000,     
                    193.482000,
                    203.000000,	212.512000,	216.845000,	220.803000,     
                    224.759000,
                    228.712000,	232.663000,	236.612000,	240.558000,     
                    245.514000,
                    252.411000,	259.303000,	266.191000,	271.150000,     
                    271.150000,
                    271.150000,	265.321000,	259.435000,	253.554000,     
                    247.675000,
                    241.801000,	235.930000,	230.063000,	224.199000,     
                    218.339000,
                    212.483000,	206.631000,	200.782000,	194.937000,     
                    189.095000,
                    190.448000 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = hdt20k1p[i];
                        density210c[i] = hd20k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double hd20k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.169000e+000,	9.639000e-001,	7.886000e-001,	
                    6.401000e-001,	5.152000e-001,
                    4.180000e-001,	3.345000e-001,	2.634000e-001,	
                    2.036000e-001,	1.455000e-001,
                    9.795000e-002,	6.710000e-002,	4.711000e-002,	
                    3.377000e-002,	2.437000e-002,
                    1.770000e-002,	1.294000e-002,	9.519000e-003,	
                    7.033000e-003,	5.186000e-003,
                    3.857000e-003,	2.892000e-003,	2.185000e-003,	
                    1.664000e-003,	1.303000e-003,
                    1.020000e-003,	8.106000e-004,	6.442000e-004,	
                    5.096000e-004,	4.012000e-004,
                    3.142000e-004,	2.448000e-004,	1.897000e-004,	
                    1.461000e-004,	1.118000e-004,
                    8.514000e-005,	6.468000e-005,	4.893000e-005,	
                    3.662000e-005,	2.710000e-005,
                    1.911000e-005 }; 

                static double hdt20k10p[NUMPROFILEPTS] = {  /* in K */ 
                    301.150000,	290.182000,	279.221000,	268.267000,    
                    257.319000,
                    241.483000,	225.580000,	209.687000,	193.803000,    
                    189.211000,
                    198.134000,	207.052000,	214.014000,	218.369000,    
                    222.720000,
                    227.069000,	231.415000,	235.758000,	240.363000,    
                    247.264000,
                    254.161000,	261.053000,	267.941000,	274.650000,    
                    274.650000,
                    274.650000,	270.610000,	265.116000,	259.627000,    
                    254.140000,
                    248.657000,	243.178000,	237.702000,	232.229000,    
                    226.760000,
                    221.294000,	214.624000,	206.826000,	199.032000,    
                    191.243000,
                    191.549000 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = hdt20k10p[i];
                        density210c[i] = hd20k10p[i];
                    }
                  }
                break;
            } break;
          case K30:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double hd30k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.248000e+000,	1.007000e+000,	8.080000e-001,	
                    6.539000e-001,	5.229000e-001,
                    4.124000e-001,	2.952000e-001,	2.180000e-001,	
                    1.623000e-001,	1.209000e-001,
                    9.007000e-002,	6.686000e-002,	4.970000e-002,	
                    3.700000e-002,	2.758000e-002,
                    2.059000e-002,	1.527000e-002,	1.127000e-002,	
                    8.388000e-003,	6.280000e-003,
                    4.732000e-003,	3.587000e-003,	2.735000e-003,	
                    2.098000e-003,	1.633000e-003,
                    1.276000e-003,	1.001000e-003,	7.900000e-004,	
                    6.266000e-004,	4.982000e-004,
                    3.952000e-004,	3.129000e-004,	2.526000e-004,	
                    2.023000e-004,	1.607000e-004,
                    1.264000e-004,	9.983000e-005,	7.811000e-005,	
                    6.003000e-005,	4.518000e-005,
                    3.318000e-005 }; 

                static double hdt30k1p[NUMPROFILEPTS] = {  /* in K */ 
                    282.150000,	273.132000,	264.103000,	250.093000,    
                    236.091000,
                    222.177000,	229.368000,	231.150000,	231.150000,    
                    231.150000,
                    231.150000,	232.137000,	233.133000,	234.127000,    
                    235.121000,
                    236.114000,	239.208000,	244.764000,	250.315000,    
                    255.864000,
                    261.409000,	266.950000,	272.488000,	278.023000,    
                    280.401000,
                    282.375000,	283.650000,	283.650000,	281.939000,    
                    278.985000,
                    276.033000,	272.927000,	263.099000,	253.277000,    
                    243.461000,
                    233.652000,	220.447000,	205.751000,	191.064000,    
                    176.386000,
                    161.717000 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = hdt30k1p[i];
                        density210c[i] = hd30k1p[i];
                    }
                  }
                break;
             case PP10:
                 {
                static double hd30k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.204000e+000,	9.916000e-001,	8.089000e-001,	
                    6.558000e-001,	5.273000e-001,
                    4.181000e-001,	3.047000e-001,	2.230000e-001,	
                    1.648000e-001,	1.226000e-001,
                    9.124000e-002,	6.790000e-002,	5.054000e-002,	
                    3.731000e-002,	2.756000e-002,
                    2.043000e-002,	1.518000e-002,	1.126000e-002,	
                    8.405000e-003,	6.313000e-003,
                    4.770000e-003,	3.626000e-003,	2.771000e-003,	
                    2.129000e-003,	1.662000e-003,
                    1.308000e-003,	1.029000e-003,	8.156000e-004,	
                    6.478000e-004,	5.132000e-004,
                    4.054000e-004,	3.194000e-004,	2.546000e-004,	
                    2.045000e-004,	1.626000e-004,
                    1.277000e-004,	9.908000e-005,	7.574000e-005,	
                    5.697000e-005,	4.208000e-005,
                    3.045000e-005 }; 

                static double hdt30k10p[NUMPROFILEPTS] = {  /* in K */ 
                    292.150000,	279.333000,	266.524000,	252.718000,    
                    237.926000,
                    223.151000,	225.946000,	228.739000,	230.150000,    
                    230.150000,
                    230.150000,	230.150000,	230.150000,	232.306000,    
                    235.285000,
                    238.263000,	241.710000,	247.064000,	252.414000,    
                    257.760000,
                    263.104000,	268.444000,	273.780000,	279.114000,    
                    281.150000,
                    281.150000,	281.150000,	279.221000,	275.873000,    
                    272.528000,
                    269.185000,	265.843000,	258.340000,	246.562000,    
                    234.791000,
                    223.027000,	211.271000,	199.522000,	187.780000,    
                    176.046000,
                    164.319000 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = hdt30k10p[i];
                        density210c[i] = hd30k10p[i];
                    }
                 }
                break;
            } break;
          case K40:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double hd40k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.196000e+000,	9.874000e-001,	8.075000e-001,	
                    6.572000e-001,	5.309000e-001,
                    4.225000e-001,	3.066000e-001,	2.236000e-001,	
                    1.645000e-001,	1.218000e-001,
                    9.041000e-002,	6.716000e-002,	4.997000e-002,	
                    3.723000e-002,	2.758000e-002,
                    2.038000e-002,	1.514000e-002,	1.132000e-002,	
                    8.515000e-003,	6.437000e-003,
                    4.891000e-003,	3.735000e-003,	2.867000e-003,	
                    2.210000e-003,	1.722000e-003,
                    1.357000e-003,	1.077000e-003,	8.574000e-004,	
                    6.805000e-004,	5.385000e-004,
                    4.249000e-004,	3.343000e-004,	2.641000e-004,	
                    2.120000e-004,	1.684000e-004,
                    1.322000e-004,	1.025000e-004,	7.833000e-005,	
                    5.888000e-005,	4.346000e-005,
                    3.141000e-005 }; 

                static double hdt40k1p[NUMPROFILEPTS] = {  /* in K */ 
                    294.150000,	280.932000,	267.723000,	253.116000,    
                    237.124000,
                    221.151000,	224.346000,	227.538000,	229.643000,    
                    230.639000,
                    231.635000,	232.630000,	233.625000,	234.619000,    
                    237.367000,
                    242.131000,	246.893000,	251.651000,	256.407000,    
                    261.159000,
                    265.909000,	270.655000,	275.399000,	280.140000,    
                    283.150000,
                    283.150000,	281.054000,	277.508000,	273.963000,    
                    270.421000,
                    266.881000,	263.343000,	257.840000,	246.062000,    
                    234.291000,
                    222.527000,	210.771000,	199.022000,	187.280000,    
                    175.546000,
                    163.819000 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = hdt40k1p[i];
                        density210c[i] = hd40k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double hd40k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.213000e+000,	9.945000e-001,	8.084000e-001,	
                    6.544000e-001,	5.261000e-001,
                    4.172000e-001,	3.041000e-001,	2.225000e-001,	
                    1.644000e-001,	1.223000e-001,
                    9.104000e-002,	6.775000e-002,	5.030000e-002,	
                    3.709000e-002,	2.746000e-002,
                    2.040000e-002,	1.514000e-002,	1.124000e-002,	
                    8.401000e-003,	6.319000e-003,
                    4.782000e-003,	3.641000e-003,	2.788000e-003,	
                    2.146000e-003,	1.686000e-003,
                    1.328000e-003,	1.054000e-003,	8.368000e-004,	
                    6.625000e-004,	5.233000e-004,
                    4.124000e-004,	3.243000e-004,	2.585000e-004,	
                    2.078000e-004,	1.653000e-004,
                    1.300000e-004,	1.009000e-004,	7.730000e-005,	
                    5.823000e-005,	4.309000e-005,
                    3.123000e-005 }; 

                static double hdt40k10p[NUMPROFILEPTS] = {  /* in K */ 
                    290.150000,	278.134000,	266.125000,	252.718000,    
                    237.926000,
                    223.151000,	225.946000,	228.739000,	230.150000,    
                    230.150000,
                    230.150000,	230.150000,	230.779000,	233.562000,    
                    236.343000,
                    239.122000,	243.150000,	248.701000,	254.249000,    
                    259.794000,
                    265.335000,	270.873000,	276.407000,	281.938000,    
                    282.650000,
                    282.650000,	280.320000,	277.561000,	274.805000,    
                    272.049000,
                    269.296000,	266.544000,	259.340000,	247.562000,    
                    235.791000,
                    224.027000,	212.271000,	200.522000,	188.780000,    
                    177.046000,
                    165.319000 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = hdt40k10p[i];
                        density210c[i] = hd40k10p[i];
                    }
                  }
                break;
            } break;
        } break;
      case LOWDENSITY:
        switch (udata->palt){
          case K5:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double ld5k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.239000e+000,	9.881000e-001,	7.617000e-001,	
                    6.285000e-001,	5.125000e-001,
                    4.123000e-001,	3.267000e-001,	2.455000e-001,	
                    1.770000e-001,	1.268000e-001,
                    9.067000e-002,	6.510000e-002,	4.697000e-002,	
                    3.404000e-002,	2.478000e-002,
                    1.812000e-002,	1.331000e-002,	9.815000e-003,	
                    7.195000e-003,	5.310000e-003,
                    3.949000e-003,	2.957000e-003,	2.229000e-003,	
                    1.692000e-003,	1.307000e-003,
                    1.032000e-003,	8.186000e-004,	6.458000e-004,	
                    5.068000e-004,	3.955000e-004,
                    3.068000e-004,	2.365000e-004,	1.812000e-004,	
                    1.378000e-004,	1.041000e-004,
                    7.803000e-005,	5.802000e-005,	4.277000e-005,	
                    3.125000e-005,	2.261000e-005,
                    1.620000e-005 }; 

                static double ldt5k1p[NUMPROFILEPTS] = {  /* in K */ 
                    281.650000,	276.647000,	280.148000,	264.162000,    
                    248.187000,
                    232.221000,	216.265000,	208.150000,	208.150000,    
                    209.592000,
                    212.577000,	215.559000,	218.539000,	221.518000,    
                    224.495000,
                    227.470000,	230.443000,	233.414000,	239.116000,    
                    245.051000,
                    250.982000,	256.910000,	262.833000,	268.753000,    
                    271.150000,
                    267.717000,	261.808000,	255.903000,	250.002000,    
                    244.104000,
                    238.209000,	232.319000,	226.432000,	220.548000,    
                    214.669000,
                    208.793000,	202.920000,	197.052000,	191.187000,    
                    185.325000,
                    179.468000 }; 
                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ldt5k1p[i];
                        density210c[i] = ld5k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double ld5k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.219000e+000,	9.641000e-001,	7.757000e-001,	
                    6.395000e-001,	5.263000e-001,
                    4.272000e-001,	3.411000e-001,	2.571000e-001,	
                    1.883000e-001,	1.349000e-001,
                    9.376000e-002,	6.661000e-002,	4.821000e-002,	
                    3.505000e-002,	2.558000e-002,
                    1.875000e-002,	1.380000e-002,	1.019000e-002,	
                    7.558000e-003,	5.668000e-003,
                    4.126000e-003,	3.080000e-003,	2.315000e-003,	
                    1.752000e-003,	1.355000e-003,
                    1.057000e-003,	8.321000e-004,	6.523000e-004,	
                    5.095000e-004,	3.965000e-004,
                    3.074000e-004,	2.393000e-004,	1.857000e-004,	
                    1.431000e-004,	1.093000e-004,
                    8.282000e-005,	6.143000e-005,	4.495000e-005,	
                    3.278000e-005,	2.382000e-005,
                    1.725000e-005 }; 

                static double ldt5k10p[NUMPROFILEPTS] = {  /* in K */ 
                    289.150000,	289.158000,	283.170000,	268.554000,    
                    251.011000,
                    233.479000,	215.958000,	207.831000,	203.650000,    
                    203.650000,
                    211.317000,	216.912000,	219.689000,	222.464000,    
                    225.238000,
                    228.009000,	230.779000,	233.548000,	236.314000,    
                    241.855000,
                    247.776000,	253.693000,	259.607000,	265.516000,    
                    266.650000,
                    265.534000,	261.601000,	257.671000,	253.744000,    
                    249.818000,
                    245.896000,	239.836000,	232.784000,	225.737000,    
                    218.694000,
                    211.655000,	207.307000,	204.963000,	202.622000,    
                    200.281000,
                    197.942000 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ldt5k10p[i];
                        density210c[i] = ld5k10p[i];
                    }
                  }
                break;
            } break;
          case K10:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double ld10k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.545000e+000,	1.143000e+000,	8.676000e-001,	
                    6.200000e-001,	4.597000e-001,
                    3.413000e-001,	2.548000e-001,	1.908000e-001,	
                    1.440000e-001,	1.083000e-001,
                    8.112000e-002,	6.052000e-002,	4.497000e-002,	
                    3.327000e-002,	2.421000e-002,
                    1.761000e-002,	1.281000e-002,	9.328000e-003,	
                    6.790000e-003,	4.943000e-003,
                    3.548000e-003,	2.558000e-003,	1.856000e-003,	
                    1.354000e-003,	9.939000e-004,
                    7.332000e-004,	5.438000e-004,	4.053000e-004,	
                    3.039000e-004,	2.337000e-004,
                    1.792000e-004,	1.360000e-004,	1.024000e-004,	
                    7.735000e-005,	5.856000e-005,
                    4.445000e-005,	3.404000e-005,	2.629000e-005,	
                    2.027000e-005,	1.559000e-005,
                    1.197000e-005 }; 

                static double ldt10k1p[NUMPROFILEPTS] = {  /* in K */ 
                    228.150000,	229.142000,	223.165000,	231.152000,    
                    232.151000,
                    233.150000,	233.150000,	232.364000,	229.174000,    
                    225.986000,
                    222.799000,	219.615000,	216.433000,	213.253000,    
                    213.150000,
                    213.150000,	213.150000,	213.150000,	213.150000,    
                    213.150000,
                    216.777000,	220.732000,	224.685000,	228.636000,    
                    232.583000,
                    236.529000,	240.472000,	244.412000,	248.030000,    
                    245.668000,
                    243.308000,	243.250000,	245.410000,	247.570000,    
                    249.728000,
                    251.884000,	252.342000,	250.384000,	248.427000,    
                    246.471000,
                    244.517000 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ldt10k1p[i];
                        density210c[i] = ld10k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double ld10k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.448000e+000,	1.102000e+000,	8.601000e-001,	
                    6.639000e-001,	4.838000e-001,
                    3.538000e-001,	2.609000e-001,	1.935000e-001,	
                    1.441000e-001,	1.076000e-001,
                    8.018000e-002,	5.957000e-002,	4.415000e-002,	
                    3.264000e-002,	2.397000e-002,
                    1.752000e-002,	1.280000e-002,	9.360000e-003,	
                    6.696000e-003,	4.826000e-003,
                    3.508000e-003,	2.571000e-003,	1.899000e-003,	
                    1.413000e-003,	1.059000e-003,
                    8.132000e-004,	6.264000e-004,	4.843000e-004,	
                    3.766000e-004,	2.921000e-004,
                    2.258000e-004,	1.741000e-004,	1.338000e-004,	
                    1.025000e-004,	7.800000e-005,
                    5.924000e-005,	4.491000e-005,	3.398000e-005,	
                    2.566000e-005,	1.934000e-005,
                    1.455000e-005 }; 

                static double ldt10k10p[NUMPROFILEPTS] = {  /* in K */ 
                    243.150000,	241.338000,	231.732000,	222.154000,    
                    224.553000,
                    226.950000,	228.150000,	228.150000,	227.165000,    
                    225.173000,
                    223.182000,	221.192000,	219.203000,	217.215000,    
                    216.150000,
                    216.150000,	216.150000,	216.150000,	221.710000,    
                    227.650000,
                    233.587000,	239.520000,	245.449000,	251.374000,    
                    257.296000,
                    258.150000,	258.150000,	257.206000,	254.252000,    
                    251.301000,
                    248.351000,	245.403000,	242.457000,	239.768000,    
                    238.002000,
                    236.238000,	234.474000,	232.712000,	230.951000,    
                    229.191000,
                    227.432000 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ldt10k10p[i];
                        density210c[i] = ld10k10p[i];
                    }
                  }
                break;
            } break;
          case K20:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double ld20k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.502000e+000,	1.093000e+000,	8.690000e-001,	
                    6.808000e-001,	5.052000e-001,
                    3.608000e-001,	2.620000e-001,	1.903000e-001,	
                    1.362000e-001,	9.686000e-002,
                    6.741000e-002,	4.882000e-002,	3.763000e-002,	
                    2.891000e-002,	2.215000e-002,
                    1.691000e-002,	1.277000e-002,	9.615000e-003,	
                    7.236000e-003,	5.447000e-003,
                    4.100000e-003,	3.087000e-003,	2.311000e-003,	
                    1.720000e-003,	1.286000e-003,
                    9.663000e-004,	7.303000e-004,	5.339000e-004,	
                    4.349000e-004,	3.369000e-004,
                    2.602000e-004,	2.003000e-004,	1.537000e-004,	
                    1.168000e-004,	8.855000e-005,
                    6.703000e-005,	5.081000e-005,	3.849000e-005,	
                    2.909000e-005,	2.194000e-005,
                    1.651000e-005 }; 

                static double ldt20k1p[NUMPROFILEPTS] = {  /* in K */ 
                    235.150000,	243.632000,	229.623000,	215.623000,    
                    210.154000,
                    213.150000,	213.150000,	213.150000,	216.598000,    
                    223.572000,
                    239.403000,	250.961000,	247.978000,	244.997000,    
                    242.017000,
                    239.040000,	238.150000,	238.150000,	238.150000,    
                    238.150000,
                    238.150000,	238.150000,	239.685000,	243.636000,    
                    247.583000,
                    251.529000,	255.150000,	255.150000,	252.590000,    
                    249.441000,
                    246.295000,	243.150000,	240.007000,	238.437000,    
                    237.456000,
                    236.475000,	234.842000,	232.884000,	230.927000,    
                    228.971000,
                    227.017000 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ldt20k1p[i];
                        density210c[i] = ld20k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double ld20k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.537000e+000,	1.125000e+000,	8.600000e-001,	
                    6.627000e-001,	5.045000e-001,
                    3.639000e-001,	2.626000e-001,	1.926000e-001,	
                    1.397000e-001,	9.979000e-002,
                    7.125000e-002,	5.000000e-002,	3.482000e-002,	
                    2.458000e-002,	1.757000e-002,
                    1.270000e-002,	9.288000e-003,	6.885000e-003,	
                    5.213000e-003,	3.961000e-003,
                    3.019000e-003,	2.315000e-003,	1.801000e-003,	
                    1.400000e-003,	1.086000e-003,
                    8.419000e-004,	6.515000e-004,	5.034000e-004,	
                    3.885000e-004,	2.993000e-004,
                    2.317000e-004,	1.810000e-004,	1.405000e-004,	
                    1.074000e-004,	8.021000e-005,
                    5.986000e-005,	4.493000e-005,	3.373000e-005,	
                    2.527000e-005,	1.888000e-005,
                    1.408000e-005 }; 

                static double ldt20k10p[NUMPROFILEPTS] = {  /* in K */ 
                    230.150000,	235.137000,	229.114000,	219.105000,    
                    209.150000,
                    209.150000,	209.137000,	205.143000,	202.150000,    
                    202.150000,
                    202.150000,	206.061000,	214.023000,	221.980000,    
                    229.932000,
                    237.879000,	245.821000,	252.753000,	255.727000,    
                    258.700000,
                    261.761000,	263.889000,	262.306000,	260.725000,    
                    259.144000,
                    257.565000,	255.986000,	254.409000,	252.832000,    
                    251.256000,
                    248.101000,	241.217000,	234.336000,	229.150000,    
                    229.150000,
                    229.150000,	227.781000,	225.821000,	223.862000,    
                    221.905000,
                    219.949000 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ldt20k10p[i];
                        density210c[i] = ld20k10p[i];
                    }
                  }
                break;
            } break;
          case K30:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double ld30k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.424000e+000,	1.076000e+000,	8.577000e-001,	
                    6.749000e-001,	5.236000e-001,
                    3.881000e-001,	2.810000e-001,	2.032000e-001,	
                    1.463000e-001,	1.048000e-001,
                    7.433000e-002,	5.172000e-002,	3.613000e-002,	
                    2.534000e-002,	1.773000e-002,
                    1.226000e-002,	8.602000e-003,	6.109000e-003,	
                    4.390000e-003,	3.190000e-003,
                    2.342000e-003,	1.736000e-003,	1.317000e-003,	
                    1.009000e-003,	7.760000e-004,
                    5.999000e-004,	4.639000e-004,	3.661000e-004,	
                    2.882000e-004,	2.255000e-004,
                    1.752000e-004,	1.329000e-004,	1.001000e-004,	
                    7.534000e-005,	5.659000e-005,
                    4.244000e-005,	3.192000e-005,	2.398000e-005,	
                    1.794000e-005,	1.338000e-005,
                    9.939000e-006 }; 

                static double ldt30k1p[NUMPROFILEPTS] = {  /* in K */ 
                    248.150000,	250.624000,	237.606000,	224.597000,    
                    211.595000,
                    205.150000,	202.892000,	199.898000,	196.905000,    
                    193.914000,
                    192.133000,	194.125000,	196.115000,	198.104000,    
                    201.374000,
                    209.121000,	216.863000,	224.601000,	232.334000,    
                    240.062000,
                    247.785000,	255.504000,	259.189000,	260.771000,    
                    262.150000,
                    262.150000,	262.150000,	256.305000,	249.408000,    
                    242.516000,
                    235.628000,	233.069000,	231.693000,	230.318000,    
                    228.944000,
                    227.570000,	225.109000,	222.170000,	219.233000,    
                    216.297000,
                    213.363000, }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ldt30k1p[i];
                        density210c[i] = ld30k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double ld30k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.430000e+000,	1.093000e+000,	8.613000e-001,	
                    6.707000e-001,	5.158000e-001,
                    3.772000e-001,	2.755000e-001,	2.003000e-001,	
                    1.450000e-001,	1.044000e-001,
                    7.490000e-002,	5.344000e-002,	3.701000e-002,	
                    2.576000e-002,	1.789000e-002,
                    1.244000e-002,	8.750000e-003,	6.218000e-003,	
                    4.463000e-003,	3.233000e-003,
                    2.363000e-003,	1.742000e-003,	1.300000e-003,	
                    9.764000e-004,	7.374000e-004,
                    5.682000e-004,	4.389000e-004,	3.440000e-004,	
                    2.709000e-004,	2.120000e-004,
                    1.648000e-004,	1.252000e-004,	9.452000e-005,	
                    7.124000e-005,	5.362000e-005,
                    4.029000e-005,	3.037000e-005,	2.287000e-005,	
                    1.715000e-005,	1.281000e-005,
                    9.539000e-006 }; 

                static double ldt30k10p[NUMPROFILEPTS] = {  /* in K */ 
                    247.150000,	245.528000,	234.312000,	223.104000, 
                    211.983000,
                    209.139000,	206.142000,	203.148000,	200.155000,    
                    197.164000,
                    194.175000,	191.188000,	193.901000,	196.686000,    
                    201.164000,
                    207.521000,	213.874000,	220.223000,	226.568000,    
                    232.909000,
                    239.246000,	245.579000,	250.648000,	255.590000,    
                    260.528000,
                    261.150000,	261.150000,	257.172000,	250.472000,    
                    243.777000,
                    237.085000,	234.569000,	233.193000,	231.818000,    
                    230.444000,
                    229.070000,	226.473000,	223.338000,	220.205000,    
                    217.074000,
                    213.944000 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ldt30k10p[i];
                        density210c[i] = ld30k10p[i];
                    }
                  }
                break;
            } break;
          case K40:
            switch (udata->ppercent) {
              case PP1:
                  {
                static double ld40k1p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.398000e+000,	1.052000e+000,	8.468000e-001,	
                    6.726000e-001,	5.266000e-001,
                    4.055000e-001,	2.888000e-001,	2.079000e-001,	
                    1.495000e-001,	1.070000e-001,
                    7.616000e-002,	5.315000e-002,	3.710000e-002,	
                    2.581000e-002,	1.782000e-002,
                    1.237000e-002,	8.646000e-003,	5.945000e-003,	
                    4.118000e-003,	2.901000e-003,
                    2.075000e-003,	1.504000e-003,	1.105000e-003,	
                    8.218000e-004,	6.177000e-004,
                    4.748000e-004,	3.757000e-004,	2.992000e-004,	
                    2.376000e-004,	1.880000e-004,
                    1.503000e-004,	1.206000e-004,	9.584000e-005,	
                    7.536000e-005,	5.857000e-005,
                    4.496000e-005,	3.328000e-005,	2.386000e-005,	
                    1.711000e-005,	1.227000e-005,
                    8.804000e-006 }; 

                static double ldt40k1p[NUMPROFILEPTS] = {  /* in K */ 
                    253.150000,	257.122000,	243.103000,	229.093000,    
                    215.091000,
                    201.150000,	201.150000,	198.747000,	195.555000,    
                    192.365000,
                    189.176000,	189.150000,	189.150000,	189.831000,    
                    192.813000,
                    195.793000,	198.771000,	206.639000,	216.553000,   
                    226.460000,
                    236.362000,	246.257000,	256.147000,	266.030000,    
                    275.907000,
                    282.150000,	280.753000,	276.810000,	272.869000,    
                    268.930000,
                    261.222000,	250.404000,	239.594000,	228.790000,    
                    217.993000,
                    207.202000,	201.150000,	201.150000,	201.150000,    
                    201.150000,
                    201.150000 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ldt40k1p[i];
                        density210c[i] = ld40k1p[i];
                    }
                  }
                break;
              case PP10:
                  {
                static double ld40k10p[NUMPROFILEPTS] = {  /* in kg/m^3 */ 
                    1.395000e+000,	1.054000e+000,	8.421000e-001,	
                    6.649000e-001,	5.185000e-001,
                    3.987000e-001,	2.922000e-001,	2.128000e-001,	
                    1.540000e-001,	1.108000e-001,
                    7.919000e-002,	5.522000e-002,	3.851000e-002,	
                    2.653000e-002,	1.818000e-002,
                    1.259000e-002,	8.805000e-003,	6.216000e-003,	
                    4.388000e-003,	3.108000e-003,
                    2.234000e-003,	1.627000e-003,	1.200000e-003,	
                    8.963000e-004,	6.761000e-004,
                    5.293000e-004,	4.187000e-004,	3.338000e-004,	
                    2.652000e-004,	2.098000e-004,
                    1.653000e-004,	1.307000e-004,	1.045000e-004,	
                    8.265000e-005,	6.456000e-005,
                    4.975000e-005,	3.642000e-005,	2.617000e-005,	
                    1.881000e-005,	1.352000e-005,
                    9.728000e-006 }; 

                static double ldt40k10p[NUMPROFILEPTS] = {  /* in K */ 
                    253.650000,	256.626000,	244.610000,	232.601000,    
                    220.599000,
                    208.635000,	204.640000,	200.647000,	196.656000,    
                    192.668000,
                    188.683000,	188.650000,	188.650000,	191.226000,    
                    196.593000,
                    201.957000,	207.317000,	212.674000,	220.053000,    
                    229.960000,
                    239.862000,	249.757000,	259.647000,	269.530000,    
                    279.407000,
                    280.650000,	278.974000,	274.242000,	269.513000,    
                    264.786000,
                    260.063000,	253.382000,	241.589000,	229.803000,    
                    218.024000,
                    206.252000,	202.650000,	202.650000,     202.650000,    
                    202.650000,
                    202.650000 }; 

                for ( i = 0; i < NUMPROFILEPTS; i++ )
                    {
                        temperature210c[i] = ldt40k10p[i];
                        density210c[i] = ld40k10p[i];
                    }
                  }
                break;
            }break;
        }break;
      default:
        break;
    }
}

