/* Copyright 2007-2014 The MathWorks, Inc. */

#include "xil_data_stream.h"
#include "codeinstr_data_stream.h"
#include "xil_interface_lib.h"
#include "comms_interface.h"
#include "xilservice.h"

#ifdef USING_CS_API
    #include "coder/connectivity/xilservice/target/CInterface.h"
    #define XIL_TX_BUFFER_MEMUNIT_SIZE commsGetMaxPayloadSize()
    #define BUFFER_HEADER_SIZE 0    
#else
    #include "rx_tx_buffer_sizes.h"    
    #define xilServiceTgtCreate()  XILSERVICE_SUCCESS
    #define xilServiceTgtDestroy()
#endif

#define USED_WRITE_BUFFER_SIZE (COMMAND_COMPLETE_SIZE + xilWriteDataAvail)
#define WRITE_BUFFER_SIZE (XIL_TX_BUFFER_MEMUNIT_SIZE - COMMAND_COMPLETE_SIZE - BUFFER_HEADER_SIZE)

static void* pBuffer;
static IOUnit_T* xilWriteBuffer;
static IOUnit_T* xilWriteDataPtr;
static uint16_T xilWriteDataAvail;
static MemUnit_T* commandResponseType;

/* reset the write buffer */
static XIL_INTERFACE_LIB_ERROR_CODE resetXILWriteBuffer(void) {
    
    /* Ask the CS to allocate a buffer that the XIL app service will use for
     * transmission
     */
    if (!commsAllocXILBuffer(&pBuffer, XIL_TX_BUFFER_MEMUNIT_SIZE)) {
        return XIL_INTERFACE_LIB_ERROR;
    }
    xilWriteBuffer = commsGetXILBufferDataPtr(pBuffer);
    
    /* set commandResponseType and xilWriteDataPtr pointer */
    commandResponseType = (MemUnit_T *) &xilWriteBuffer[COMMAND_COMPLETE_IDX];
    xilWriteDataPtr = &xilWriteBuffer[WRITE_DATA_BUFFER_IDX];
    
    /* ready for next command */
    xilWriteDataAvail = 0;
    *commandResponseType = XIL_COMMAND_NOT_COMPLETE;
    
    return XIL_INTERFACE_LIB_SUCCESS;
}

XIL_INTERFACE_LIB_ERROR_CODE xilInit(const int argc,
        void *argv[]) {
    
    /* Create comm service */
    if (commsCreate(argc, argv) == COMMS_ERROR) {
        return XIL_INTERFACE_LIB_ERROR;
    }
    
    /* Create XIL service */    
    if (xilServiceTgtCreate() == XILSERVICE_ERROR) {
        return XIL_INTERFACE_LIB_ERROR;
    }
    
    /* Create code instrumentation service if required */
#ifdef CODE_INSTRUMENTATION_ENABLED
    if (codeInstrInit()!= XIL_INTERFACE_LIB_SUCCESS) {
        return XIL_INTERFACE_LIB_ERROR;
    }
#endif
    
   /* Reset write buffer */
   if (resetXILWriteBuffer()!= XIL_INTERFACE_LIB_SUCCESS) {
	   return XIL_INTERFACE_LIB_ERROR;
   }
    
    return XIL_INTERFACE_LIB_SUCCESS;
}

/* This function must be called prior to terminating the application in order to
 * ensure an orderly shutdown of communications - data will already have been sent */
XIL_INTERFACE_LIB_ERROR_CODE xilTerminateComms(void) {      
   /* Destroy XIL service */
   xilServiceTgtDestroy();
   
   /* destroy code instrumentation service */
#ifdef CODE_INSTRUMENTATION_ENABLED
   codeInstrTerminate();
#endif
   
   /* Destroy comm service */
   commsDestroy();    
   
   return XIL_INTERFACE_LIB_SUCCESS;
}

/* send pending writes */
static XIL_DATA_STREAM_ERROR_CODE sendWriteBuffer(void) {
   /* send */
   if (commsEnqueueBuffer(pBuffer, XIL_SERVICE_ID, USED_WRITE_BUFFER_SIZE) == COMMS_ERROR) {             
      return XIL_DATA_FLUSH_ERROR;
   }   
   
   /* reset */
   if (resetXILWriteBuffer()!= XIL_INTERFACE_LIB_SUCCESS) {
	   return XIL_DATA_FLUSH_ERROR;
   }
   
   return XIL_DATA_STREAM_SUCCESS;
}


XIL_DATA_STREAM_ERROR_CODE xilWriteData(const MemUnit_T * src, uint32_T size) {
   XIL_DATA_STREAM_ERROR_CODE errorCode = XIL_DATA_STREAM_SUCCESS;
   const IOUnit_T * srcPtr = (const IOUnit_T *) src;   
   size_t transferAmount;
   uint16_T bufferAvail;
  
   /* block until all data is processed */
   while (size > 0) {      
      /* send if we have a full message worth of data */   
      if (xilWriteDataAvail == WRITE_BUFFER_SIZE) {
         errorCode = sendWriteBuffer();
         if (errorCode != XIL_DATA_STREAM_SUCCESS) {
            return errorCode;
         }
      }
      bufferAvail = WRITE_BUFFER_SIZE - xilWriteDataAvail;
      transferAmount = (uint16_T) MIN(bufferAvail, size);
      /* copy data into write buffer */
      memcpy((void *) xilWriteDataPtr, srcPtr, transferAmount);
      size -= (uint32_T) transferAmount;
      xilWriteDataAvail += (uint16_T) transferAmount;
      srcPtr += transferAmount;
      xilWriteDataPtr += transferAmount;
   }
   return errorCode;
}


XIL_DATA_STREAM_ERROR_CODE xilDataFlush(MemUnit_T isEndOfXILStep) {
    
    /* final part of command */
    if (isEndOfXILStep) {
        *commandResponseType = XIL_STEP_COMPLETE;
    } else {
        *commandResponseType = XIL_COMMAND_COMPLETE;
    }

    /* send the write buffer */
    if (sendWriteBuffer() == XIL_DATA_FLUSH_ERROR){
        return XIL_DATA_FLUSH_ERROR;
    }
    
    return XIL_DATA_STREAM_SUCCESS;

}
