/*
 * File: aeroatmnonstd.c
 *
 * Abstract:
 *    Function to calculate non-standard day atmosphere 
 *
 *     
 * Copyright 1990-2013 The MathWorks, Inc.
 *
 * Author:  S. Gage      15-MAR-2006
 */

#include <math.h>
#include <stddef.h>
#include "export/include/aeroatmos/aeroatm.hpp"
#include "export/include/aeroatmos/aeroatmnonstd.hpp"
#include "export/include/aeroatmos/aeroatmosstruct.hpp"
#include "export/include/aeroatmos/aerointerpolate.hpp"

/* Function: CalcAtmosEnvelope ============================================
 * Abstract:
 *   Using cached tables, find the working interval and 
 *   perform logarithmic interpolation.
 */

void CalcAtmosEnvelope(const double *altitude,
                       const double *altitudeTab,
                       const double *temperatureTab,
                       const double *densityTab,
                       const double *pressureTab, 
                       double *temp, 
                       double *pressure, 
                       double *density,
                       double *speedofsound,
                       const size_t numPoints,
                       const SFcnCache *udata,
		       double *minmax)
{
    size_t i;

    for (i=0; i < numPoints; i++) {
        double fraction = 0.0; 
        unsigned int idx;

        /* Find altitude interval using binary search */
        idx = LookBinXd( altitude[i], &fraction, altitudeTab, (const unsigned int) minmax[0] ); 

        /* Interval has been obtained, now do interpolation */

        switch (udata->epercent) {
          case EXTREME:
          case P1:
            switch (udata->evar) {
              case HIGHTEMP:
              case LOWTEMP:
              case HIGHDENSITY:
              case LOWDENSITY:
                {
                    /* linear interpolation for temperature and 
                     * log interpolation for density 
                     */

                    InterpolateTempAndDens( altitude[i], temperatureTab, 
                                            densityTab, &temp[i], &density[i], 
                                            fraction, idx, minmax[1], 
                                            minmax[2] );

                    pressure[i] = density[i]*((R_HAT/MOL_WT)*temp[i]);
                    speedofsound[i] = sqrt(GAMMA*temp[i]*(R_HAT/MOL_WT));
                }
                break;
              case HIGHPRESSURE:
              case LOWPRESSURE:
                {
                    /* 
                     * log interpolation for pressure 
                     */

                    AltitudeLimitLogInterpolate(altitude[i], pressureTab,
                                                fraction, &pressure[i], idx, minmax[1],
                                                minmax[2]);
                    /* not enough information to calculate remaining data*/
                    temp[i] = 0.0;
                    density[i] = 0.0;
                    speedofsound[i] = 0.0;
                }
                break;
            }
            break;
          case P5:
          case P10:
          case P20:
            switch (udata->evar) {
              case HIGHTEMP:
              case LOWTEMP:
                {
                    AltitudeLimitLinearInterpolate(altitude[i], temperatureTab,
                                                   fraction, &temp[i], idx, minmax[1],
                                                   minmax[2]);
                    /* not enough information to calculate remaining data */
                    density[i] = 0.0;
                    pressure[i] = 0.0;
                    speedofsound[i] = 0.0;
                }
                break;
              case HIGHDENSITY:
              case LOWDENSITY:
                {
                    /* 
                     * log interpolation for density 
                     */
                    AltitudeLimitLogInterpolate(altitude[i], densityTab,
                                                fraction, &density[i], idx, minmax[1],
                                                minmax[2]);
                    /* not enough information to calculate remaining data */
                    temp[i] = 0.0;
                    pressure[i] = 0.0;
                    speedofsound[i] = 0.0;
                }
                break;

              case HIGHPRESSURE:
              case LOWPRESSURE:
                {
                    /* 
                     * log interpolation for pressure 
                     */

                    AltitudeLimitLogInterpolate(altitude[i], pressureTab,
                                                fraction, &pressure[i], idx, minmax[1],
                                                minmax[2]);
                    /* not enough information to calculate remaining data */
                    temp[i] = 0.0;
                    density[i] = 0.0;
                    speedofsound[i] = 0.0;
                }
                break;
            }
            break;
        }
    }
}

/* Function: CalcAtmosProfile ===============================================
 * Abstract:
 *   Using cached tables, find the working interval and 
 *   perform logarithmic interpolation.
 */

void CalcAtmosProfile(const double *altitude,
                      const double *altitudeTab,
                      const double  *temperatureTab,
                      const double  *densityTab, 
                      double *temp, 
                      double *pressure, 
                      double *density,
                      double *speedofsound,
                      const size_t   numPoints,
                      const double  maxIdx,
                      const double  lower_alt,
                      const double  upper_alt  )
{
    size_t i;
    
    for (i=0; i < numPoints; i++) {
        double fraction = 0.0; 
        unsigned int idx;

        /* Find altitude interval using binary search */
        idx = LookBinXd( altitude[i], &fraction, altitudeTab, (const unsigned int)maxIdx ); 


        /* Interval has been obtained, now do interpolation */

        InterpolateTempAndDens( altitude[i], temperatureTab, 
                                densityTab, &temp[i], &density[i], 
                                fraction, idx, lower_alt, upper_alt );

        pressure[i] = density[i]*((R_HAT/MOL_WT)*temp[i]);
        speedofsound[i] = sqrt(GAMMA*temp[i]*(R_HAT/MOL_WT));

    }
}
